#-------------------------------------------------------------------------------------------------#
library(lavaan)                                                  # activate                       #
library(psych)                                                   # desired packages               #
#-------------------------------------------------------------------------------------------------#
#                                                                # set working directory          #
setwd('D:/MyStuff/cls/gradquant/example02_pathanalysis02')               
#-------------------------------------------------------------------------------------------------#

cmat  = as.matrix(read.table(file = 'educ_attain.cor.txt'))
print(cmat)
sdvec = as.matrix(read.table(file = 'educ_attain.sd.txt' ))
print(sdvec)

sdmat = diag(5)
diag(sdmat) = sdvec
print(sdmat)

covs  =  sdmat %*% cmat %*% sdmat
print(covs)

rownames(covs) = c('ma','sesr','ap','so','ea')
colnames(covs) = c('ma','sesr','ap','so','ea')

#-------------------------------------------------------------------------------------------------#
#  manifest variable       latent variable         definition
#
#  ma                      mentabil                mental ability
#  sesr                    ses                     socioeconomic status
#  ap                      acadperf                academic performance (class rank)
#  so                      sigother                significant other influence
#  ea                      edaspire                educational aspirations
#-------------------------------------------------------------------------------------------------#               

#-------------------------------------------------------------------------------------------------#
# Path Model #1: Sewell et al. model as "manifest variable" model
#-------------------------------------------------------------------------------------------------#
path1.mv = 'ap ~ ma
            so ~ ma + sesr + ap
            ea ~ ap + so
            
            ma   ~~ ma
            sesr ~~ sesr
            sesr ~~ ma
            ap   ~~ ap
            so   ~~ so
            ea   ~~ ea '
     
fit.path1.mv = lavaan(model=path1.mv, sample.cov = covs, sample.nobs=4388, fixed.x=FALSE)

summary             (fit.path1.mv, standardized=TRUE, fit.measures=TRUE, rsquare=TRUE)
parameterEstimates  (fit.path1.mv)
inspect             (fit.path1.mv)

#-------------------------------------------------------------------------------------------------#
# Path Model #2: Sewell et al. model as "latent variable" model, but assume "perfect measurement"
#   This just makes explicit what is implicit in the preceding "simple" syntax and makes very
#   clear the assumption of perfect measurement, so each latent variable is equated with its 
#   manifest indicator in a "warts and all" (i.e., total variance) fashion
#-------------------------------------------------------------------------------------------------#
path2.lv = '
          # loadings of MVs on LVs in Lambda (L)  # lavaan code =~
            mentabil =~ 1*ma
            ses      =~ 1*sesr
            acadperf =~ 1*ap
            sigother =~ 1*so
            edaspire =~ 1*ea

          # intercepts of MVs in Nu (N)           # lavaan code ~
            # no means are used in this model

          # unique vars & covs of MVs in Theta (T)# lavaan code ~~
            ma   ~~ 0*ma
            sesr ~~ 0*sesr
            ap   ~~ 0*ap
            so   ~~ 0*so
            ea   ~~ 0*ea 

          # means of LVs in Alpha (A)             # lavaan code ~
            # again, no means used in this model

          # vars & covariances of LVs in Psi (P)  # lavaan code ~~
            mentabil ~~ mentabil
            ses      ~~ ses
            ses      ~~ mentabil
            acadperf ~~ acadperf
            sigother ~~ sigother
            edaspire ~~ edaspire

          # regressions among LVs in Beta (B)     # lavaan code ~
            acadperf ~ mentabil
            sigother ~ mentabil + ses + acadperf
            edaspire ~                  acadperf + sigother

            '
          
fit.path2.lv = lavaan(model=path2.lv, sample.cov = covs, sample.nobs=4388, fixed.x=FALSE)

summary             (fit.path2.lv, standardized=TRUE, fit.measures=TRUE, rsquare=TRUE)
parameterEstimates  (fit.path2.lv)
inspect             (fit.path2.lv)
    
#-------------------------------------------------------------------------------------------------#
# Path Model #3: Sewell et al. model as "latent variable" model, with unreliability excluded
#
# Estimates of reliablity (rxx) for manifest variables, lead to estimates of unreliable variance
#   Remember KR20 and KR 21 -- the 20th and 21st formulas in Kuder & Richardson (1937)? Well,
#   KW1 = first thing that popped into Keith Widaman's head when considering manifest variables
#   JH2 = second thoughts on reliability by Jonathan Helm (use these for ap and so)
#
#          KW1     JH2     Unreliable variance = (1 - rxx) * variance of manifest variable
#   ma     .94     same    estimate for ma     = (1 - .94) * 850.597 =  51.0358
#   sesr   .85     same    estimate for sesr   = (1 - .85) * 122.944 =  18.4416
#   ap     .90     .70     estimate for ap     = (1 - .70) * 731.703 = 219.5108
#   so     .70     .62     estimate for so     = (1 - .62) *   2.924 =   1.1112
#   ea     .64     same    estimate for ma     = (1 - .64) *   0.887 =   0.3195
#-------------------------------------------------------------------------------------------------#
path3.lv = '
          # loadings of MVs on LVs in Lambda (L)  # lavaan code =~
            mentabil =~ 1*ma
            ses      =~ 1*sesr
            acadperf =~ 1*ap
            sigother =~ 1*so
            edaspire =~ 1*ea

          # intercepts of MVs in Nu (N)           # lavaan code ~
            # no means are used in this model

          # unique vars & covs of MVs in Theta (T)# lavaan code ~~
            ma   ~~  51.0358*ma
            sesr ~~  18.4416*sesr
            ap   ~~ 219.5108*ap
            so   ~~   1.1112*so
            ea   ~~   0.3195*ea 

          # means of LVs in Alpha (A)             # lavaan code ~
            # again, no means used in this model

          # vars & covariances of LVs in Psi (P)  # lavaan code ~~
            mentabil ~~ start(799.34)*mentabil
            ses      ~~ start(104.47)*ses
            ses      ~~ start(093.36)*mentabil
            acadperf ~~ start(241.51)*acadperf
            sigother ~~ start(000.64)*sigother
            edaspire ~~ start(000.02)*edaspire

          # regressions among LVs in Beta (B)     # lavaan code ~
            acadperf ~ start(0.582)*mentabil
            sigother ~ start(0.001)*mentabil + start(0.046)*ses + start(0.038)*acadperf
            edaspire ~ start(-.002)*acadperf + start(0.577)*sigother
            '
          
fit.path3.lv = lavaan(model=path3.lv, sample.cov = covs, sample.nobs=4388, fixed.x=FALSE)

summary             (fit.path3.lv, standardized=TRUE, fit.measures=TRUE, rsquare=TRUE)
parameterEstimates  (fit.path3.lv)
inspect             (fit.path3.lv)    

#-------------------------------------------------------------------------------------------------#
# Path Model #4: Sewell et al. model, reliability excluded, and model trimmed
#-------------------------------------------------------------------------------------------------#
path4.lv = '
          # loadings of MVs on LVs in Lambda (L)  # lavaan code =~
            mentabil =~ 1*ma
            ses      =~ 1*sesr
            acadperf =~ 1*ap
            sigother =~ 1*so
            edaspire =~ 1*ea

          # intercepts of MVs in Nu (N)           # lavaan code ~
            # no means are used in this model

          # unique vars & covs of MVs in Theta (T)# lavaan code ~~
            ma   ~~  51.0358*ma
            sesr ~~  18.4416*sesr
            ap   ~~ 219.5108*ap
            so   ~~   1.1112*so
            ea   ~~   0.3195*ea 

          # means of LVs in Alpha (A)             # lavaan code ~
            # again, no means used in this model

          # vars & covariances of LVs in Psi (P)  # lavaan code ~~
            mentabil ~~ start(799.34)*mentabil
            ses      ~~ start(104.47)*ses
            ses      ~~ start(093.36)*mentabil
            acadperf ~~ start(241.51)*acadperf
            sigother ~~ start(000.64)*sigother
            edaspire ~~ start(000.02)*edaspire

          # regressions among LVs in Beta (B)     # lavaan code ~
            acadperf ~ start(0.582)*mentabil
            sigother ~ start(0.046)*ses + start(0.038)*acadperf
            edaspire ~ start(0.577)*sigother
            '
          
fit.path4.lv = lavaan(model=path4.lv, sample.cov = covs, sample.nobs=4388, fixed.x=FALSE)

summary             (fit.path4.lv, standardized=TRUE, fit.measures=TRUE, rsquare=TRUE)
parameterEstimates  (fit.path4.lv)
inspect             (fit.path4.lv)


 

 