#-------------------------------------------------------------------------------------------------#
library(lavaan)                                                  # activate                       #
library(psych)                                                   # desired packages               #
#-------------------------------------------------------------------------------------------------#
#                                                                # set working directory          #
setwd('D:/MyStuff/cls/gradquant/example03_cfa')               
#-------------------------------------------------------------------------------------------------#

hs39raw = read.table(file = 'hs39all_raw.txt', na = '.')

#-------------------------------------------------------------------------------------------------#
# NOTE: I just read in the "raw-scored" version of the Holzinger-Swineford 1939 data              #
#-------------------------------------------------------------------------------------------------#
head(hs39raw)
dim(hs39raw)

names(hs39raw) = c('case'    , 'school'  , 'grade'   , 'female'  , 'ageyr'  , 'mo'      , 'agemo', 
                   'visperc' , 'cubes'   , 'frmbord' , 'lozenges', 'geninfo', 'paracomp',
                   'sentcomp', 'wordclas', 'wordmean', 'addition', 'code'   , 'countdot', 
                   'sccaps'  , 'wordrecg', 'numbrecg', 'figrrecg', 'objnumb', 'numbfig' , 
                   'figword' , 'deducton', 'numbpuzz', 'probreas', 'series' , 'woody'   , 
                   'frmbord2', 'flags'   )
head(hs39raw)

#-------------------------------------------------------------------------------------------------#
#  latent variable     manifest variable     definition of manifest variable        
#
#  Spatial (S)         visperc    x1         
#  Spatial (S)         cubes      x2         similarity of 3-D cubes
#  Spatial (S)         lozenges   x3         2 vs 3 dimensional rotation of parallelograms
#  Verbal (V)          paracomp   x4         paragraph commprehension
#  Verbal (V)          sentcomp   x5         sentence comprehension
#  Verbal (V)          wordmean   x6         word meaning / vocabulary
#  PercSpeed (P)       addition   x7         addition problems
#  PercSpeed (P)       countdots  x8         counting dots in a visual array
#  PercSpeed (P)       sccaps     x9         straight vs curved capital letters
#-------------------------------------------------------------------------------------------------#  

#-------------------------------------------------------------------------------------------------#
# CFA model 0: using the HolzingerSwineford1939 data set "sitting around" when you invoke lavaan
#   This version of the data set has transformed scores, not the original raw scores
#-------------------------------------------------------------------------------------------------#
holz = HolzingerSwineford1939
dim(holz)

hs.model0 = '
            spatial  =~ x1  + x2  + x3
            verbal   =~ x4  + x5  + x6
            speed    =~ x7  + x8  + x9
            '

fit0 = cfa(hs.model0, data = holz)

summary(fit0, fit.measures = TRUE, standardized = TRUE)
inspect(fit0)


#-------------------------------------------------------------------------------------------------#
# CFA model 1: use raw data from Holzinger and Swineford (1939) -- same a priori model
#-------------------------------------------------------------------------------------------------#

hs.model1 = '
           #loadings of MVs on LVs in Lambda (L)
            spatial  =~ visperc  + cubes    + lozenges
            verbal   =~ paracomp + sentcomp + wordmean
            speed    =~ addition + countdot + sccaps
            '

fit1 = cfa(hs.model1, data = hs39raw)

summary(fit1, fit.measures = TRUE, standardized = TRUE)
inspect(fit1)
MI1 = modificationIndices(fit1)
subset(MI1, mi > 10)


#-------------------------------------------------------------------------------------------------#
# CFA model 2: modify model to let sccaps load on Spatial factor
#-------------------------------------------------------------------------------------------------#

hs.model2 = '
           #loadings of MVs on LVs in Lambda (L)
            spatial  =~ visperc  + cubes    + lozenges 
            verbal   =~ paracomp + sentcomp + wordmean
            speed    =~ addition + countdot + sccaps
            spatial  =~ sccaps                             # here is the added loading
            '

fit2 = cfa(hs.model2, data = hs39raw)

summary(fit2, fit.measures = TRUE, standardized = TRUE)
inspect(fit2)
MI2 = modificationIndices(fit2)
subset(MI2, mi > 5)


#-------------------------------------------------------------------------------------------------#
# CFA model 3: same as model 2, but expanded syntax explicit about all parts of the CFA model
#-------------------------------------------------------------------------------------------------#

hs.model3 = '
          # loadings of MVs on LVs in Lambda (L)  # lavaan code =~
            spatial  =~ NA*visperc  + cubes    + lozenges 
            verbal   =~ NA*paracomp + sentcomp + wordmean
            speed    =~ NA*addition + countdot + sccaps
            spatial  =~ sccaps                             # here is the added loading

          # intercepts of MVs in Nu (N)           # lavaan code ~
            visperc  ~ NA*1  ;  cubes    ~ NA*1  ;  lozenges ~ NA*1
            paracomp ~ NA*1  ;  sentcomp ~ NA*1  ;  wordmean ~ NA*1
            addition ~ NA*1  ;  countdot ~ NA*1  ;  sccaps   ~ NA*1

          # unique vars & covs of MVs in Theta (T)# lavaan code ~~
            visperc  ~~ NA*visperc   ;  cubes ~~ NA*cubes
            lozenges ~~ NA*lozenges
            paracomp ~~ NA*paracomp  ;  sentcomp ~~ NA*sentcomp
            wordmean ~~ NA*wordmean
            addition ~~ NA*addition  ;  countdot ~~ NA*countdot
            sccaps   ~~ NA*sccaps

          # means of LVs in Alpha (A)             # lavaan code ~
            spatial ~ 0
            verbal  ~ 0
            speed   ~ 0

          # vars & covariances of LVs in Psi (P)  # lavaan code ~~
            spatial ~~ 1*spatial
            verbal  ~~ 1*verbal
            speed   ~~ 1*speed

            verbal  ~~ spatial
            speed   ~~ spatial  ;  speed ~~ verbal

          # regressions among LVs in Beta (B)     # lavaan code ~

            '

fit3 = cfa(hs.model3, data = hs39raw)

summary(fit3, fit.measures = TRUE, standardized = TRUE)
inspect(fit3)
MI3 = modificationIndices(fit3)
subset(MI3, mi > 5)


#-------------------------------------------------------------------------------------------------#
# CFA model 4: same as model 3, but use the lavaan command, rather than cfa
#-------------------------------------------------------------------------------------------------#

hs.model4 = '
          # loadings of MVs on LVs in Lambda (L):   lavaan code =~
            spatial  =~ NA*visperc  + cubes    + lozenges 
            verbal   =~ NA*paracomp + sentcomp + wordmean
            speed    =~ NA*addition + countdot + sccaps
            spatial  =~ sccaps                             # here is the added loading

          # intercepts of MVs in Nu (N):            lavaan code ~
            visperc  ~ NA*1  ;  cubes    ~ NA*1  ;  lozenges ~ NA*1
            paracomp ~ NA*1  ;  sentcomp ~ NA*1  ;  wordmean ~ NA*1
            addition ~ NA*1  ;  countdot ~ NA*1  ;  sccaps   ~ NA*1

          # unique vars & covs of MVs in Theta (T): lavaan code ~~
            visperc  ~~ NA*visperc   ;  cubes    ~~ NA*cubes
            lozenges ~~ NA*lozenges
            paracomp ~~ NA*paracomp  ;  sentcomp ~~ NA*sentcomp
            wordmean ~~ NA*wordmean
            addition ~~ NA*addition  ;  countdot ~~ NA*countdot
            sccaps   ~~ NA*sccaps

          # means of LVs in Alpha (A):              lavaan code ~
            spatial ~ 0*1
            verbal  ~ 0*1
            speed   ~ 0*1

          # vars & covariances of LVs in Psi (P):   lavaan code ~~
            spatial ~~ 1*spatial
            verbal  ~~ 1*verbal
            speed   ~~ 1*speed

            verbal  ~~ spatial
            speed   ~~ spatial  ;  speed ~~ verbal

          # regressions among LVs in Beta (B):      lavaan code ~
            '

fit4 = lavaan(model = hs.model4, data = hs39raw, 
              int.ov.free = TRUE , int.lv.free = TRUE,
              std.ov      = FALSE, std.lv      = FALSE)

summary(fit4, fit.measures = TRUE, standardized = TRUE)
inspect(fit4)
MI4 = modificationIndices(fit4)
subset(MI4, mi > 5)


 